//
// example code for async read/write of stdin/out <-> com port
//

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <windows.h>
#include <X11/Xlib.h>

/*
  gcc -O -Wall -g hoge.c -o hoge.exe -mwindows -mwinmm -lX11
*/

#include "mini4WD2packet.h"
struct ctrlpacket_struct cpkt;
struct senspacket_struct spkt;

/****************************************************************/
/*	UART							*/

#define ComPortNo 3

// memo : configuration of com port on windows
// mode.com com1: BAUD=115200 PARITY=N DATA=8 STOP=1 to=off xon=off odsr=off octs=on dtr=off rts=on idsr=off

FILE *logfp,*recfp;
int Logfile_Init(){
  time_t t;
  struct tm *t_local;
  char buf[32];
  time(&t);
  t_local=localtime(&t);

  sprintf(buf,"log_%02d%02d%02d_%02d%02d%02d.txt",
	  (t_local->tm_year)%100,t_local->tm_mon+1,t_local->tm_mday,
	  t_local->tm_hour,t_local->tm_min,t_local->tm_sec);
  logfp=fopen(buf,"w");
  if (logfp==NULL) {
    fprintf(stderr,"can't open log file (%s)\n",buf);
    exit(1);
  }
  printf("Log file (%s) ready\n",buf);
  fprintf(logfp,"Logging starts at %02d/%02d/%02d %02d:%02d:%02d\n",
	  (t_local->tm_year)%100,t_local->tm_mon+1,t_local->tm_mday,
	  t_local->tm_hour,t_local->tm_min,t_local->tm_sec);

  sprintf(buf,"rec_%02d%02d%02d_%02d%02d%02d.dat",
	  (t_local->tm_year)%100,t_local->tm_mon+1,t_local->tm_mday,
	  t_local->tm_hour,t_local->tm_min,t_local->tm_sec);
  recfp=fopen(buf,"wb");
  if (recfp==NULL) {
    fprintf(stderr,"can't open record file (%s)\n",buf);
    exit(1);
  }
  printf("Record file (%s) ready\n",buf);

  return 1;
}

int UART_Init(int p){
  static int fd;
  static struct termios term;
  char buf[32];
  sprintf(buf,"/dev/ttyS%d",p-1); // example COM1 -> /dev/ttyS0
  fd = open(buf,O_RDWR);
  if(fd<0){
    fprintf(stderr,"COM%d open error\n",p);
    exit(1);
  }

  tcgetattr(fd,&term);
  /* flag &=~(a|b|..|z) ... disables the listed properties */
  /* flag |=  a|b|..|z  ... enables the listed properties */
  term.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON|IXOFF);
  term.c_cflag &= ~(CSIZE|CSTOPB|PARENB);
  term.c_cflag |= CS8;
  term.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  term.c_cc[VMIN]=0;
  term.c_cc[VTIME]=1;
  cfsetispeed(&term,B115200);
  cfsetospeed(&term,B115200);

  tcflush(fd,TCIFLUSH);
  tcsetattr(fd,TCSANOW,&term);

  printf("COM%d (%s) ready\n",p,buf);

  return fd;
}

/****************************************************************/
/*	X11							*/

#define WIN_W    1024  /* EBhE̕   */
#define WIN_H     512  /* EBhE̍ */

Display* x_dpy;
int      x_scr;
Window   x_win0;   /* root window */
Window   x_win;    /* drawing windows */
GC       x_gc;     /* Graphics Context */
Pixmap   x_img1,x_img2; /* image buffers */
XEvent   x_evt;
int      g_skip=32;  /* max shift per win update (pixel) */
int      g_count=0;  /* num drawn (pixel) */
DWORD    g_period=500; /* period of win update (msec) */

int Xwin_Init(void){
  XWindowAttributes atr;
  x_dpy  = XOpenDisplay(":0");
  if (x_dpy==NULL) {
    fprintf(stderr,"can't open X11 display!\n");
    exit(1);
  }
  x_win0 = DefaultRootWindow(x_dpy);
  x_scr  = DefaultScreen(x_dpy);
  x_win  = XCreateSimpleWindow
    (x_dpy,x_win0,0,0,WIN_W,WIN_H,2,0x808080,0x000000);
  XGetWindowAttributes(x_dpy,x_win,&atr);
  x_img1 = XCreatePixmap(x_dpy,x_win,WIN_W,WIN_H,atr.depth);
  x_img2 = XCreatePixmap(x_dpy,x_win,WIN_W,WIN_H,atr.depth);
  x_gc   = XCreateGC(x_dpy,x_win,0,NULL);
  XSelectInput(x_dpy,x_win,KeyPressMask|ExposureMask);
  XMapWindow(x_dpy,x_win);
  XFlush(x_dpy);

  printf("X11 window system ready\n");
  return 1;
}

/****************************************************************/
/*	Joypad							*/

JOYINFOEX joypad;
int joypad_X0,joypad_Y0,joypad_Z0,joypad_R0;
#define joypad_stickmargin 0x2000

int Joypad_Init(void){
  joypad.dwSize = sizeof(JOYINFOEX);
  joypad.dwFlags = JOY_RETURNALL;
  if (JOYERR_NOERROR!=joyGetPosEx(0,&joypad)) {
    fprintf(stderr,"joypad error\n");
    exit(1);
  }
  printf("joypad0 ready\n");
  joypad_X0=joypad.dwXpos;
  joypad_Y0=joypad.dwYpos;
  joypad_Z0=joypad.dwZpos;
  joypad_R0=joypad.dwRpos;
  printf("center values of (X,Y,Z,R)=(%08x,%08x,%08x,%08x)\n",
	 joypad_X0,joypad_Y0,joypad_Z0,joypad_R0);

  return 1;
}


/****************************************************************/
/*	misc							*/

int absmax_vec(int v[],int len){
  int max,x;
  max=0;
  for (int i=0;i<len;i++) {
    x=v[i];
    if (x<0) x=-x;
    if (x>max) max=x;
  }
  return max;
}

int max2(int x, int y){
  if (x>y)
    return x;
  else
    return y;
}

/****************************************************************/
/*	main							*/

int main(int argc, char **argv){

  Logfile_Init();

  int portnum = ComPortNo;
  if (argc>=2) portnum=atoi(argv[1]);
  int uartfd = UART_Init(portnum);

  init_ctrlpacket(&cpkt);
  init_senspacket(&spkt);

  Xwin_Init();

  Joypad_Init();
  JOYINFOEX joypad_prev;
  joyGetPosEx(0,&joypad_prev);

  DWORD t0,t1; /* last sent & last drawn (msec) */
  t0=t1=timeGetTime();

  int rcvcount=0;
  unsigned char tx1=0,tx2=0,tx3=0;
  unsigned char mtr_on=0;
  int mag_max=1, acl_max=1, gyr_max=1, pd_max=1, thm_max=1;
  int run=1;
  while (run) {
    int n;

    DWORD t=timeGetTime();  /* msec */

    // read joypad
    if (JOYERR_NOERROR!=joyGetPosEx(0,&joypad)) {
      fprintf(stderr,"Joypad Error\n");
      exit(1);
    }
    if (joypad.dwButtons&0x050)  run=0; // quit if button 5 or 7
    if (joypad.dwButtonNumber>3) run=0; // quit if 4+ button pressed

    if((joypad.dwButtons&0x300)||// push left or right bar
       (joypad.dwYpos>joypad_Y0+joypad_stickmargin)|| // push down left  bar
       (joypad.dwRpos>joypad_R0+joypad_stickmargin)){ // push down right bar
      cpkt.brk=1;
      mtr_on=0;
    } else if (joypad.dwButtons&0x024){ // button 3 or 6
      cpkt.brk=1;
    } else {
      cpkt.brk=0;
    }

    if (cpkt.brk) {
      cpkt.mtr=0xff; // no power
    } else if (joypad.dwYpos<joypad_Y0-joypad_stickmargin) {
      cpkt.mtr=(joypad.dwYpos>>7)&0xff; // map Y(0000,7FFF) to (00,FF)
      mtr_on=0;
    } else if (joypad.dwRpos<joypad_R0-joypad_stickmargin) {
      cpkt.mtr=(joypad.dwRpos>>7)&0xff; // map R(0000,7FFF) to (00,FF)
      mtr_on=0;
    } else if (joypad.dwButtons&0x008){ // button 4
      cpkt.mtr=0;    // full power
      mtr_on=0;
    } else if (joypad.dwButtons&0x080){ // button 8
      cpkt.mtr=0;    // full power
      mtr_on=1;
    } else if (mtr_on) {
      cpkt.mtr=0;    // full power
    } else {
      cpkt.mtr=0xff; // no power
    }

    if ((0xffff&joypad_prev.dwPOV)==0xffff && (0xffff&joypad.dwPOV)!=0xffff) {
      /* mode by POV */
      switch (0xffff&joypad.dwPOV) {
      case     0:
	if (cpkt.rate<15) cpkt.rate++;
	break;
      case  9000:
	mtr_on=1;
	break;
      case 18000:
	if (cpkt.rate> 0) cpkt.rate--;
	break;
      case 27000:
	mtr_on=0;
	break;
      }
      printf("sampling %ldHz\n",1000000/ratecode2period(cpkt.rate));
      fprintf(logfp,"sampling %ldHz\n",1000000/ratecode2period(cpkt.rate));
      printf("motor mode No.%d\n",mtr_on);
    }
    cpkt.opt1=(joypad.dwButtons&0x001)?1:0;
    cpkt.opt2=(joypad.dwButtons&0x002)?1:0;
    cpkt.opt3=(joypad.dwButtons&0x800)?1:0;
    joypad_prev=joypad;

    if (run==0) {
      cpkt.brk=0;
      cpkt.mtr=0xff;
      cpkt.rate=0;
      cpkt.opt3=1;
    }
    
    pack_ctrlpacket(&cpkt);
    if ((cpkt.packet[1]!=tx1)||(cpkt.packet[2]!=tx2)||(cpkt.packet[3]!=tx3)||
	t-t0>250){
      n=write(uartfd,cpkt.packet,CTRLPKTSZ);
      if (n!=CTRLPKTSZ) {
	fprintf(stderr,"UART write error\n");
	exit(1);
      }
      t0=timeGetTime();
    }
    tx1=cpkt.packet[1];
    tx2=cpkt.packet[2];
    tx3=cpkt.packet[3];
    
    while (1) {
      // read uart, non-blocking, multi-bytes
      n=read(uartfd,spkt.packet+rcvcount,SENSPKTSZ-rcvcount);
      if (n<0) {
	fprintf(stderr,"UART read error\n");
	exit(1);
      }
      rcvcount=rcvcount+n;
      if (rcvcount<SENSPKTSZ) break;
      if (spkt.packet[0]!=0x55) {
	for (int i=1;i<rcvcount;i++)
	  spkt.packet[i-1]=spkt.packet[i];
	rcvcount--;
	fprintf(logfp,"no sync header, skip byte\n");
	continue;
      }
      rcvcount=0;
      fwrite(spkt.packet,1,SENSPKTSZ,recfp);
      unpack_senspacket(&spkt);

      pd_max =max2(absmax_vec(&(spkt.pd ),1), pd_max);
      mag_max=max2(absmax_vec(  spkt.mag, 3),mag_max);
      acl_max=max2(absmax_vec(  spkt.acl, 3),acl_max);
      gyr_max=max2(absmax_vec(  spkt.gyr, 3),gyr_max);
      thm_max=max2(absmax_vec(&(spkt.thm),1),thm_max);
      
      float scale;
      int x,y;
      x=WIN_W-g_skip+g_count;
      
      /* Photo Diode */
      scale=0.8*((float)WIN_H)/((float)pd_max);
      y=WIN_H-1-((int)(((float)spkt.pd)*scale));
      XSetForeground(x_dpy,x_gc, 0x808080 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      
      /* Magnetic */
      scale=0.4*((float)WIN_H)/((float)mag_max);
      y=WIN_H/2-((int)(((float)spkt.mag[0])*scale));
      XSetForeground(x_dpy,x_gc, 0xff0000 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.mag[1])*scale));
      XSetForeground(x_dpy,x_gc, 0xff8000 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.mag[2])*scale));
      XSetForeground(x_dpy,x_gc, 0xff0080 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      
      /* Accel */
      scale=0.4*((float)WIN_H)/((float)acl_max);
      y=WIN_H/2-((int)(((float)spkt.acl[0])*scale));
      XSetForeground(x_dpy,x_gc, 0x00ff00 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.acl[1])*scale));
      XSetForeground(x_dpy,x_gc, 0x80ff00 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.acl[2])*scale));
      XSetForeground(x_dpy,x_gc, 0x00ff80 );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      
      /* Gyro */
      scale=0.4*((float)WIN_H)/((float)gyr_max);
      y=WIN_H/2-((int)(((float)spkt.gyr[0])*scale));
      XSetForeground(x_dpy,x_gc, 0x0000ff );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.gyr[1])*scale));
      XSetForeground(x_dpy,x_gc, 0x8000ff );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      y=WIN_H/2-((int)(((float)spkt.gyr[2])*scale));
      XSetForeground(x_dpy,x_gc, 0x0080ff );
      XDrawPoint(x_dpy,x_img1,x_gc,x,y);
      
      g_count++;
      if (g_count>=g_skip) break;

    }

    // update window & monitoring
    if ((g_count>0 && t-t1>g_period) || g_count>=g_skip) {
      XCopyArea(x_dpy,x_img1,x_win, x_gc,      0,0,WIN_W        ,WIN_H,0,0);
      XCopyArea(x_dpy,x_img1,x_img2,x_gc,      0,0,WIN_W        ,WIN_H,0,0);
      XCopyArea(x_dpy,x_img2,x_img1,x_gc,g_count,0,WIN_W-g_count,WIN_H,0,0);
      XSetForeground(x_dpy,x_gc, 0x000000 );
      XFillRectangle(x_dpy,x_img1,x_gc,WIN_W-g_skip,0,WIN_W-1,WIN_H);
      XFlush(x_dpy);

      printf("%6ldms ",spkt.t/1000);
      printf("STA%02x BTN%02x VOL%02x ",spkt.stat,spkt.sw,spkt.vol);
      printf("BRK%d MTR%02x ",spkt.brk,spkt.mtr);
      printf("PD%4d ",spkt.pd);
      printf("MAG%5d%5d%5d ",spkt.mag[0],spkt.mag[1],spkt.mag[2]);
      printf("ACL%5d%5d%5d ",spkt.acl[0],spkt.acl[1],spkt.acl[2]);
      printf("GYR%5d%5d%5d ",spkt.gyr[0],spkt.gyr[1],spkt.gyr[2]);
      printf("THM%3d ",spkt.thm);
      printf("\n");

      t1=timeGetTime();
      g_count=0;
    }
    
    if (XCheckWindowEvent(x_dpy,x_win,KeyPressMask,&x_evt)) {
      switch(x_evt.type) {
      case KeyPress:
	run=0;
	break;
      }
    }

    usleep(1000);
  }

  XCloseDisplay(x_dpy);
  close(uartfd);
  fclose(recfp);
  fclose(logfp);
  
  return 0;
}
